﻿-- Hear Kitty  by Vger-Azjol-Nerub
-- www.vgermods.com
-- 
-- Version 1.1: configuration UI, support for multiple queued sounds, support for Ruthlessness (bonus point on finishing move)
--
------------------------------------------------------------

-- Hear Kitty requires this version of VgerCore:
local KittyVgerCoreVersionRequired = 1.02

-- "Constants"
local KittySoundDelay = 0.35
local KittyLongSoundDelay = 1 -- for transitioning from 0 to 1

-- Timer
local KittyTimerCountdown = 0

-- Other
local KittyLastSoundPlayed = 0
local KittyQueueActive = false
local KittySoundOnTimer = nil
local KittyRequeuesOnTimer = 0


------------------------------------------------------------
-- Hear Kitty events
------------------------------------------------------------

-- Called when an event that Hear Kitty cares about is fired.
function KittyOnEvent(Event)
	if Event == "PLAYER_COMBO_POINTS" then
		KittyOnComboPointsChange()
	elseif Event == "VARIABLES_LOADED" then 
		KittyInitialize()
	end 
end

-- Initializes Hear Kitty after all saved variables have been loaded.
function KittyInitialize()

	-- Check the current version of VgerCore.
	if (not VgerCore) or (not VgerCore.Version) or (VgerCore.Version < KittyVgerCoreVersionRequired) then
		if DEFAULT_CHAT_FRAME then DEFAULT_CHAT_FRAME:AddMessage("|cfffe8460" .. KittyLocal.NeedNewerVgerCoreMessage) end
		message(KittyLocal.NeedNewerVgerCoreMessage)
		return
	end

	SLASH_HEARKITTY1 = "/hearkitty"
	SlashCmdList["HEARKITTY"] = KittyCommand

	-- If they don't have any options set yet (no saved variable), reset them.  If they upgraded
	-- from a previous version and are missing one or more options, fill them in with defaults.
	KittyFillMissingOptions()
	
end

-- Called whenever the player's number of combo points changes.
function KittyOnComboPointsChange()
	if (KittyOptions.Enabled == true) then KittyComboSound(GetComboPoints()) end
end

-- Handles timer updates.  Called once per video frame.
function KittyOnUpdate(Elapsed)
	KittyTimerCountdown = KittyTimerCountdown - Elapsed
	if KittyTimerCountdown <= 0 then
		-- Was there a sound on the timer?  If not, then the timer can now be shut off.
		if KittySoundOnTimer == nil then
			KittyStopTimer()
			return
		end
		-- Play this sound effect.
		KittyPlayOneSound(KittySoundOnTimer)
		-- Then, queue the next sound if there is one, or queue the "sounds are done" timer.
		KittyTimerCountdown = KittyTimerCountdown + KittySoundDelay
		if KittyRequeuesOnTimer > 0 then
			KittySoundOnTimer = KittySoundOnTimer + 1
			KittyRequeuesOnTimer = KittyRequeuesOnTimer - 1
		else
			KittySoundOnTimer = nil
		end
	end
end

------------------------------------------------------------
-- Hear Kitty methods
------------------------------------------------------------

-- Resets all Hear Kitty options.  Used to set the saved variable to a default state.
function KittyResetOptions()
	KittyOptions = {}
	KittyFillMissingOptions()
end

-- Adds default values for any Hear Kitty options that are missing.  This can happen after an upgrade.
function KittyFillMissingOptions()
	if not KittyOptions then KittyOptions = {} end
	
	if KittyOptions.Enabled == nil then KittyOptions.Enabled = true end
	if KittyOptions.DoubleCrits == nil then KittyOptions.DoubleCrits = true end
end

-- Starts the timer counting down from a specified duration.
function KittyStartTimer(Duration)
	KittyQueueActive = true
	KittyTimerCountdown = Duration
	KittyCoreFrame:SetScript("OnUpdate", function() KittyOnUpdate(arg1) end)
end

-- Immediately stops the timer from firing.  Does not clear any other state information.
function KittyStopTimer()
	KittyQueueActive = false
	KittyTimerCountdown = nil
	KittySoundOnTimer = nil
	KittyCoreFrame:SetScript("OnUpdate", nil)
end

-- Processes a Hear Kitty slash command.
function KittyCommand(Command)
	local Lower = strlower(Command)
	if Lower == "" or Lower == nil then
		KittyUI_Show()
	elseif Lower == KittyLocal.EnabledCommand then
		KittyEnable(true)
		KittySayIfOn()
	elseif Lower == KittyLocal.DisabledCommand then
		KittyEnable(false)
		KittySayIfOn()
	else
		KittyUsage()
	end
end

-- Enables or disables Hear Kitty sound effects.
function KittyEnable(Enabled)
	VgerCore.Assert(Enabled == true or Enabled == false, "New value should be true or false.")
	KittyOptions.Enabled = Enabled
	KittyStopTimer()
end

-- Prints a message stating whether or not Hear Kitty is enabled.
function KittySayIfOn()
	if KittyOptions.Enabled then
		VgerCore.Message(VgerCore.Color.Blue .. KittyLocal.EnabledMessage)
	else
		VgerCore.Message(VgerCore.Color.Blue .. KittyLocal.DisabledMessage)
	end
end

-- Enables or disables double crit sounds.
function KittyEnableDoubleCrits(Double)
	VgerCore.Assert(Double == true or Double == false, "New value should be true or false.")
	KittyOptions.DoubleCrits = Double
end

-- Plays the appropriate sound effect for when the number of combo points changes.
-- If the number of combo points changed by +2, play one sound and then the second
-- one after a short duration.
function KittyComboSound(ComboPoints)
	VgerCore.Assert(ComboPoints >= 0 and ComboPoints <= 5, "ComboPoints is expected to be between 0 and 5.")
	
	-- If the current number of combo points is zero, but we just played that sound effect, don't play it
	-- again.  This happens in various situations, such as when deselecting a freshly-killed enemy.
	if ComboPoints == 0 and KittyLastSoundPlayed == ComboPoints then return end
	
	-- If they have the double crits option disabled, this is all very easy.  Never queue.
	if KittyOptions.DoubleCrits == false then
		KittyPlayOneSound(ComboPoints)
		return
	end
	
	-- What we do next depends on whether we've increased or decreased combo points.
	if ComboPoints < KittyLastSoundPlayed then
		-- If the number of combo points has decreased, then they must have spent some.  Play the zero
		-- sound effect.  Then, if they have more than zero, they must have also gained some, so queue
		-- additional sounds.
		KittyStopTimer()
		KittyPlaySoundRange(0, ComboPoints, KittyLongSoundDelay)
	elseif ComboPoints == KittyLastSoundPlayed then
		-- If the number of combo points stayed the same, just replay the sound if it's 1 -- this covers most
		-- cases of quickly changing targets and adding a combo point.  In other cases, we'll ignore the
		-- duplicated event.
		if ComboPoints == 1 then
			KittyPlaySoundRange(ComboPoints, ComboPoints)
		end
	else
		-- If the number of combo points increased, play the new range of sounds.  If one's already queued,
		-- we'll just queue more instead of playing anything immediately.
		KittyPlaySoundRange(KittyLastSoundPlayed + 1, ComboPoints)
	end
	
	-- Remember the last number played from this function so we can play the correct sounds next time.
	KittyLastSoundPlayed = ComboPoints
end

-- Plays a range of sounds, queueing as necessary.  Optionally specifies the duration for the first sound queue;
-- subsequent sounds will use the standard duration.
function KittyPlaySoundRange(Start, End, FirstDuration)
	VgerCore.Assert(Start <= End, "Start should be less than or equal to End.")
	if Start > End then return end
	
	-- If there are other sounds waiting in the queue, just add to the queue and don't play anything immediately.
	if KittyQueueActive then
		if KittySoundOnTimer then
			-- There are other actual sounds waiting in the queue.
			KittyRequeuesOnTimer = End - KittySoundOnTimer
		else
			-- There are no sounds waiting in the queue, but we just finished playing the last one.
			KittySoundOnTimer = Start
			KittyRequeuesOnTimer = End - Start - 1
		end
		return
	end
	
	-- Play the starting sound.
	KittyPlayOneSound(Start)
	
	-- Finally, queue other sounds as necessary.
	if Start == End then
		-- We don't need to queue other sounds, but we do need to put "nothing" on the queue so that the next sound
		-- played still has to wait its turn.
		KittySoundOnTimer = nil
		KittyRequeuesOnTimer = 0
	else
		KittySoundOnTimer = Start + 1
		KittyRequeuesOnTimer = End - Start - 1
	end
	if FirstDuration == nil then FirstDuration = KittySoundDelay end
	KittyStartTimer(FirstDuration)
end

-- Immediately plays one combo point sound effect.  (Contrast with KittyComboSound.)
function KittyPlayOneSound(ComboPoints)
	PlaySoundFile("Interface\\AddOns\\Hear Kitty\\" .. ComboPoints .. ".mp3")
end

-- Displays Hear Kitty usage information.
function KittyUsage()
	VgerCore.Message(" ")
	VgerCore.MultilineMessage(KittyLocal.Usage, VgerCore.Color.Blue)
	VgerCore.Message(" ")
end

------------------------------------------------------------

-- Core frame setup
if not KittyCoreFrame then
	KittyCoreFrame = CreateFrame("Frame", "KittyCoreFrame")
end

KittyCoreFrame:SetScript("OnEvent", function() KittyOnEvent(event, arg1, arg2, arg3) end)

KittyCoreFrame:RegisterEvent("VARIABLES_LOADED")
KittyCoreFrame:RegisterEvent("PLAYER_COMBO_POINTS")
